﻿
// test.wav ファイルの作成例で　BinaryWriter　のバイナリ書き込みを使って実現する例
// Wave wavefile = new Wave(1, 8000, 10, path);// モノラルで8KHzで10秒用のファイルを作り
// wavefile.WriteUnit16Data(UInt16のデータ);の繰り返しだけで、ファイルが生成できるようにクラス化している。


// 上記を利用して、static メソッドを作った。
// それで、static パラメタに従って、マイ・ミュージックに「umehoshiEdit.wav」に生成できる。
// WriteData0(short val, (16bitのデータ, 1)の繰り返しだけでモノラルで8KHzで10秒用の生成をできるようにした。
// Close0()で10秒に満たなくても閉じる


using System.Diagnostics;
using System;
using System.IO;
using System.Text;

namespace umehoshiEdit
{
    public class Wave
    {
        public static int SamplingRate = 8000;//WriteData0メソッド専用のサンプリングレート
        public static int GenerationSeconds = 10;//WriteData0メソッド専用の生成時間(単位：秒)
        public static Wave wavefile0 = null;// WriteData0メソッド専用の書き込みストリーム（nullで生成）

        // インスタンス変数
        private FileStream filStream;// waveファイル操作用ストリーム
        private BinaryWriter binWriter { get; set; } = null;// waveファイル操作用バイナリ書き込み操作用
        private WaveHeader header;// waveファイルのヘッダ部管理用
        private int writeCount = 0;// サインプルデータの出力数（WriteUnit16Dataの実行でカウント）

        public Wave()
        {
            //TestMake();// テスト用
            test();
        }

        //~Wave()// デストラクタ (binWriterが開放されてから実行されて、実行エラーになる)
        //{
        //    if (this.binWriter == null) return;
        //    while (this.writeCount < this.header.DataLength)
        //    {
        //        this.WriteUnit16Data(0);//指定サイズに満たない部分は0で埋める
        //    }
        //}

        // チャンネル数と、サンプリングレート でpathファイルを生成してヘッダーまでを書き込む。
        public Wave(int numbChannel, int rate, int sec, string path)
        {
            this.header = new WaveHeader(numbChannel, rate, sec);
            Debug.Print("----------データ数：" + this.header.DataLength);

            this.filStream = new FileStream(path, FileMode.Create, FileAccess.Write);//ファイル生成

            this.binWriter = new BinaryWriter(filStream);// バイナリストリーム生成

            this.binWriter.Write(this.header.Bytes);// ヘッダー部をファイルに書き込む
        }

        public void WriteUnit16Data(UInt16 data)
        {
            if (binWriter == null)
            {
                return; // 既に書き込み終了
            }
            this.binWriter.Write(BitConverter.GetBytes(data));//チャンネル１の出力

            writeCount++;
            if (writeCount >= this.header.DataLength)
            {
                this.binWriter.Close();
                this.binWriter = null;
                this.filStream.Close();

            }
        }

        // 使い方の例
        public static void test()
        {
            string path = Environment.GetFolderPath(Environment.SpecialFolder.DesktopDirectory) + @"\test.wav";

            // int numbChannel, int rate, int sec, string path の引数で生成
            Wave wavefile = new Wave(1, 8000, 10, path);

            for (UInt32 cnt = 0; cnt < wavefile.header.DataLength/2; cnt++)
            {
                double Radian = (double)cnt / wavefile.header.SamplingRate;
                Radian *= 2 * Math.PI;

                double Wave = Math.Sin(Radian * 440);// 440Hz

                UInt16 Data = (UInt16)(Wave * 65535 / 2);// 最大振幅

                wavefile.WriteUnit16Data(Data);
            }
        }

        /*
           作成するWAVEファイルは、44.1kHz16bit2chで、長さは10秒のファイル生成（ch1/ch2共に440Hz正弦波）
           インスタンス変数を使わない動作確認用
         */
        void TestMake()
        {
            string path = Environment.GetFolderPath(Environment.SpecialFolder.DesktopDirectory) + @"\test.wav";
            using (FileStream filStream = new FileStream(path, FileMode.Create, FileAccess.Write))
            {

                WaveHeader header = new WaveHeader(2, 44100, 10);

                using (BinaryWriter binWriter = new BinaryWriter(filStream))
                {
                    binWriter.Write(header.Bytes);// ヘッダー部をファイルに書き込む
                    Debug.Print("----------データ数：" + header.DataLength);

                    for (UInt32 cnt = 0; cnt < header.DataLength; cnt++)
                    {
                        double Radian = (double)cnt / header.SamplingRate;
                        Radian *= 2 * Math.PI;

                        double Wave = Math.Sin(Radian * 440);// 440Hz

                        Int16 Data = (Int16)(Wave * 65535 / 2);// 最大振幅

                        binWriter.Write(BitConverter.GetBytes(Data));//チャンネル１の出力
                        binWriter.Write(BitConverter.GetBytes(Data));//チャンネル２の出力
                    }
                }
            }
        }

        // staticパラメタで、waveファイルに書き込む（生成してなければ生成）
        // numbChannelは、1がモノラルで、2がステレオの指定
        public static void WriteData0(short val, int numbChannel)
        {
            try
            {
                //string filePath = Environment.GetFolderPath(Environment.SpecialFolder.DesktopDirectory) + @"\test.wav";
                string filePath = Environment.GetFolderPath(Environment.SpecialFolder.MyMusic) + @"\umehoshiEdit.wav";
                if (wavefile0 == null)
                {
                    wavefile0 = new Wave(numbChannel, SamplingRate, GenerationSeconds, filePath);// ファイル生成
                }

                if (wavefile0 != null)
                {
                    wavefile0.WriteUnit16Data((UInt16)(val << 5));// 増幅して書き込み
                    //wavefile.WriteUnit16Data((UInt16)(val * 2 * 2 * 2 * 2 * 2 * 2));
                }
            }
            catch (Exception e)
            {
                Console.Write(e);
                //Debug.Print(e.Message);
            }
        }

        public static void Close0()// WriteData0に対する閉じる処理
        {
            try
            {
                if (wavefile0 != null)
                {
                    wavefile0.binWriter.Close();
                    wavefile0.binWriter = null;
                    wavefile0.filStream.Close();
                    wavefile0 = null;
                }
            }
            catch (Exception e)
            {
                Console.Write(e);
                //Debug.Print(e.Message);
            }
        }

    }
    
    class WaveHeader // Waveファイルのヘッダー部(44byte) の管理
    {
        public UInt16 NumberOfChannel { get; set; } = 2;
        public UInt32 SamplingRate { get; set; } = 44100;
        public UInt16 NumberOfBitPerSample { get; set; } = 16;
        public UInt32 NumberOfBytesOfWaveData { get; set; } = 0;// ファイルのヘッダ部を除いたバイト数

        public UInt32 DataLength = 0; // ファイル内のサンプリング数

        public WaveHeader() { }//上記デフォルト値を使ったコンストラクタ

        // チャンネル数と、サンプリングレート, 生成ファイルの時間指定コンストラクタ
        public WaveHeader(int numbChannel, int rate, int sec) {
            this.NumberOfChannel = (UInt16)numbChannel;
            this.SamplingRate = (UInt16)rate;

            this.DataLength = this.SamplingRate * (UInt32)sec;//sec秒に必要なデータ数取得

            this.NumberOfBytesOfWaveData = this.BlockSize * DataLength;// ファイルのデータ部サイス
        }

        public byte[] Bytes // Waveファイルのヘッダー部(44byte) のバイト列を取得
        {
            get
            {
                byte[] Datas = new byte[44];

                Array.Copy(Encoding.ASCII.GetBytes("RIFF"), 0, Datas, 0, 4);
                Array.Copy(BitConverter.GetBytes((UInt32)(FileSize - 8)), 0, Datas, 4, 4);
                Array.Copy(Encoding.ASCII.GetBytes("WAVE"), 0, Datas, 8, 4);
                Array.Copy(Encoding.ASCII.GetBytes("fmt "), 0, Datas, 12, 4);
                Array.Copy(BitConverter.GetBytes((UInt32)(16)), 0, Datas, 16, 4);
                Array.Copy(BitConverter.GetBytes((UInt16)(1)), 0, Datas, 20, 2);
                Array.Copy(BitConverter.GetBytes((UInt16)(NumberOfChannel)), 0, Datas, 22, 2);
                Array.Copy(BitConverter.GetBytes((UInt32)(SamplingRate)), 0, Datas, 24, 4);
                Array.Copy(BitConverter.GetBytes((UInt32)(DataRate)), 0, Datas, 28, 4);
                Array.Copy(BitConverter.GetBytes((UInt16)(BlockSize)), 0, Datas, 32, 2);
                Array.Copy(BitConverter.GetBytes((UInt16)(NumberOfBitPerSample)), 0, Datas, 34, 2);
                Array.Copy(Encoding.ASCII.GetBytes("data"), 0, Datas, 36, 4);
                Array.Copy(BitConverter.GetBytes((UInt32)(NumberOfBytesOfWaveData)), 0, Datas, 40, 4);

                return (Datas);
            }
        }

        public UInt16 NumberOfBytePerSample // １サンプルで必要バイト数（デフォルトの16bitなら2が戻る）
        {
            get
            {
                return ((UInt16)(Math.Ceiling((double)NumberOfBitPerSample / 8)));
            }
        }

        public UInt32 FileSize // このヘッダを使うファイル全体のバイト数
        {
            get
            {
                return (NumberOfBytesOfWaveData + 44);
            }
        }

        public UInt32 DataRate // 「ヘッダ内パラメタの一つ」で、1秒に必要なサンプルデータのバイト数
        {
            get
            {
                return (SamplingRate * NumberOfChannel * NumberOfBytePerSample);
            }
        }

        public UInt16 BlockSize // 「ヘッダ内パラメタの一つ」で、一つサンプルデータのバイト数
        {
            get
            {
                return (UInt16)(NumberOfBytePerSample * NumberOfChannel);
            }
        }
    }
}
